﻿<?php
/**
 * MAPA DE APROVEITAMENTO ESCOLAR
 * Exibe as médias trimestrais (MT1, MT2 ou MT3) dos alunos de uma turma.
 */
session_start();
include 'conexao.php';

// =================================================================
// 0. FUNÇÕES AUXILIARES 
// =================================================================

/**
 * Corrige caracteres acentuados específicos para padronização de exibição.
 * @param string $nome O nome da disciplina.
 * @return string O nome corrigido.
 */
function corrigir_acentos($nome) {
    // Exemplo de correção simples: 'Françês' -> 'Francês'
    $map = [
        'Françês' => 'Francês',
        'Língua Nacional' => 'L. Nacional',
        // Adicionar mais correções se necessário
    ];
    return strtr($nome, $map);
}

/**
 * Normaliza o nome da disciplina para comparação e ordenação.
 * Remove acentos, espaços, pontuações e converte para maiúsculas.
 * @param string $nome O nome da disciplina.
 * @return string O nome normalizado (ex: "L.Port" -> "LPORT").
 */
function normalizar_nome($nome) {
    $nome = mb_strtoupper(trim($nome), 'UTF-8');
    $nome = preg_replace('/[.,\/ -]/', '', $nome);
    $nome = str_replace(
        ['Á', 'À', 'Ã', 'Â', 'É', 'È', 'Ê', 'Í', 'Ó', 'Õ', 'Ô', 'Ú', 'Ç'],
        ['A', 'A', 'A', 'A', 'E', 'E', 'E', 'I', 'O', 'O', 'O', 'U', 'C'],
        $nome
    );
    // Casos especiais comuns em nomes de disciplinas pedagógicas
    $nome = str_replace('EXP.MOTORA', 'EXPMOTORA', $nome);
    $nome = str_replace('EXP.PLAST', 'EXPPLAST', $nome);
    $nome = str_replace('EXP.MUSICAL', 'EXPMUSICAL', $nome);
    $nome = str_replace('MET.L.PORT', 'METLPORT', $nome);
    $nome = str_replace('MET.L.NACIONAL', 'METLNACIONAL', $nome);
    $nome = str_replace('MET.INGLES', 'METINGLES', $nome);
    $nome = str_replace('MET.L.FRANC', 'METLFRANC', $nome);

    // Mapeamento crucial: Se for o nome completo "Estudo do Meio/C.N", normaliza para a chave comum
    if (strpos($nome, 'METESTMEIOCN') !== false) {
        $nome = 'METESTMEIO';
    }
    
    return $nome;
}


// =================================================================
// 1. AUTENTICAÇÃO E CONFIGURAÇÃO
// =================================================================
const PERFIL_SUBDIRECAO = 'SUBDIRECÇÃO PEDAGÓGICA';
const PERFIL_SECRETARIA = 'SECRETARIA PEDAGÓGICA';
const CURSO_PRIMARIO_NORM = 'ENSINO PRIMÁRIO';
const CURSO_INFANCIA_NORM = 'EDUCAÇÃO DE INFÂNCIA';
const CLASSES_SECUNDARIAS = ['10ª', '11ª', '12ª', '13ª']; 

$perfil = strtoupper($_SESSION['usuario']['perfil'] ?? '');
if (!in_array($perfil, [PERFIL_SUBDIRECAO, PERFIL_SECRETARIA])) {
    header("Location: login.php");
    exit;
}

if ($conn) {
    $conn->set_charset("utf8");
}

setlocale(LC_TIME, 'pt_PT.UTF-8', 'pt_BR.UTF-8', 'portuguese');
date_default_timezone_set('Africa/Luanda');

// =================================================================
// 5. FUNÇÕES DE ORDENAÇÃO DE DISCIPLINAS (CRUCIAL)
// =================================================================

function ordenar_disciplinas($disciplinas, $classe_turma = '', $curso_turma = '') {
    // 5.1. Normaliza os nomes das disciplinas para comparação
    foreach ($disciplinas as &$d) {
        $d['nome_norm'] = normalizar_nome($d['nome']);
    }
    unset($d);

    $ordem_oficial = [];
    $curso_norm = normalizar_nome($curso_turma);
    $classe_turma = mb_strtoupper(trim($classe_turma), 'UTF-8');

    // =============================================
    // DEFINIÇÃO DA ORDEM OFICIAL BASEADA EM CLASSE/CURSO
    // =============================================
    
    // 1. REGRA MÁXIMA PRIORIDADE: 10ª Classe (Lista Curta Definida)
    if ($classe_turma === '10ª') {
        
        // CHAVES NORMALIZADAS (15 Disciplinas)
        $ordem_oficial = [
            'LPORT', 'INGLES', 'FRANCES', 'LNACIONAL', 'MAT', 'FISICA', 'BIOLOGIA',
            'QUIMICA', 'HISTORIA', 'GEOGRAFIA', 'INFORMATICA', 'EXPMUSICAL', 'EXPPLAST',
            'EXPMOTORA', 'PSICOLOGIA',
        ];
        
    // 2. REGRA PARA 11ª CLASSE INFÂNCIA (Lista especial, inclui Met.L.Infantil e Met.Est.Meio)
    } elseif ($classe_turma === '11ª' && $curso_norm === normalizar_nome(CURSO_INFANCIA_NORM)) {
        
        // LISTA 2 (Educação de Infância Padrão) - 29 Disciplinas
        $ordem_oficial = [
            'LPORT', 'INGLES', 'FRANCES', 'LNACIONAL', 'MAT', 'FISICA', 'BIOLOGIA',
            'QUIMICA', 'HISTORIA', 'GEOGRAFIA', 'INFORMATICA', 'EXPMUSICAL', 'EXPPLAST',
            'EXPMOTORA', 'PSICOLOGIA', 'FILOSOFIA', 'TEDC', 'NEE',
            'METLINFANTIL', 'METLNACIONAL', 'METESTMEIO', 'PPS1',
            'ASEAGE', 'FPSD', 'HSE', 'METEXPMOTORA', 'METEXPPLAST',
            'METMAT', 'METEXPMUSICAL', 'PPS2'
        ];
        
    // 3. REGRA PARA OUTRAS CLASSES SECUNDÁRIAS (12ª, 13ª ou 11ª Primário / 12ª Infância)
    } elseif (in_array($classe_turma, CLASSES_SECUNDARIAS) && 
              ($curso_norm === normalizar_nome(CURSO_PRIMARIO_NORM) || $curso_norm === normalizar_nome(CURSO_INFANCIA_NORM))) { 
        
        // LISTA COMPLETA (30 Disciplinas)
        $ordem_oficial = [
            'LPORT', 'INGLES', 'FRANCES', 'LNACIONAL', 'MAT', 'FISICA', 'BIOLOGIA',
            'QUIMICA', 'HISTORIA', 'GEOGRAFIA', 'INFORMATICA', 'EXPMUSICAL', 'EXPPLAST',
            'EXPMOTORA', 'PSICOLOGIA', 'FILOSOFIA', 'TEDC', 'NEE', 'METLPORT',
            'METLNACIONAL', 'METINGLES', 
            'METESTMEIO', // Usa a chave normalizada comum (graças ao mapeamento em normalizar_nome)
            'FPSD', 'HSE',
            'ASEAGE', 'METMAT', 'METHISTGEOG', 'METEXPRESSÕES', 'METLFRANC',
            'PPS1', 'PPS2'
        ];
        
    } else {
        // Fallback: nenhuma ordem oficial definida
        $ordem_oficial = []; 
    } 
    
    // 5.2. Cria o mapa de posições
    $posicoes = array_flip($ordem_oficial); 

    // 5.3. Ordena o array
    usort($disciplinas, function ($a, $b) use ($posicoes) {
        // Disciplinas não encontradas na ordem oficial (posicao 999) vão para o final
        $pos_a = $posicoes[$a['nome_norm']] ?? 999; 
        $pos_b = $posicoes[$b['nome_norm']] ?? 999;
        
        if ($pos_a === $pos_b) {
            // Se têm a mesma prioridade (na lista oficial ou no grupo '999'), ordena alfabeticamente
            return strcmp($a['nome_norm'], $b['nome_norm']); 
        }
        return $pos_a <=> $pos_b; // Coloca as não-oficiais no final
    });

    // 5.4. **IMPORTANTE:** Não filtramos mais. Retornamos TODAS as disciplinas que vieram do BD (ou da lista manual, se o BD estiver vazio).
    return $disciplinas;
}


// =================================================================
// 2. BUSCA DE DADOS INICIAIS
// =================================================================
$turmas = [];
$res = $conn->query("SELECT id, nome, classe, curso, turno, ano FROM turmas ORDER BY classe, nome");
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $turmas[] = $row;
    }
    $res->free();
}

$turma_id = filter_input(INPUT_GET, 'turma_id', FILTER_VALIDATE_INT);
$trimestre = filter_input(INPUT_GET, 'trimestre', FILTER_VALIDATE_INT);

$alunos = $disciplinas = $dados = [];
$nome_turma = $classe_turma = $curso_turma = $trimestre_extenso = '';

// =================================================================
// 3. BUSCA DE INFORMAÇÕES DA TURMA
// =================================================================
if ($turma_id) {
    $stmt_turma = $conn->prepare("SELECT nome, classe, curso FROM turmas WHERE id = ?");
    $stmt_turma->bind_param("i", $turma_id);
    $stmt_turma->execute();
    $res_turma = $stmt_turma->get_result();
    // Corrigido: $res_res_turma -> $res_turma
    if ($row_turma = $res_turma->fetch_assoc()) { 
        $nome_turma = htmlspecialchars($row_turma['nome']);
        $classe_turma = htmlspecialchars($row_turma['classe']);
        $curso_turma = htmlspecialchars($row_turma['curso']);
    }
    $stmt_turma->close();
}

if ($trimestre == 1) $trimestre_extenso = 'I TRIMESTRE';
elseif ($trimestre == 2) $trimestre_extenso = 'II TRIMESTRE';
elseif ($trimestre == 3) $trimestre_extenso = 'III TRIMESTRE';

// =================================================================
// 4. BUSCA DE ALUNOS, DISCIPLINAS E NOTAS
// =================================================================
$mapa_notas = [];

if ($turma_id && $trimestre) {
    // 4.1. Busca de Alunos
    $stmt_alunos = $conn->prepare("SELECT id, nome FROM alunos WHERE turma_id = ? ORDER BY nome");
    $stmt_alunos->bind_param("i", $turma_id);
    $stmt_alunos->execute();
    $alunos = $stmt_alunos->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_alunos->close();

    // 4.2. Busca de Disciplinas Cadastradas na Turma
    $stmt_disciplinas = $conn->prepare("SELECT id, nome FROM disciplinas WHERE turma_id = ?");
    $stmt_disciplinas->bind_param("i", $turma_id);
    $stmt_disciplinas->execute();
    $res_disciplinas_bd = $stmt_disciplinas->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_disciplinas->close();

    $disciplinas_com_id_real = !empty($res_disciplinas_bd);
    $disciplinas_para_ordenar = [];

// --- LÓGICA DE DADOS DA DISCIPLINA ---
if (!$disciplinas_com_id_real) {
    // SE NÃO HÁ DISCIPLINAS NO BD, USAMOS A LISTA MANUAL COMO PLACEHOLDER PARA ORDENAR
    $curso_norm = normalizar_nome($curso_turma);
    $classe_turma_norm = mb_strtoupper(trim($classe_turma), 'UTF-8');
    $nomes_oficiais = [];
    
    // 1. REGRA MÁXIMA PRIORIDADE: 10ª Classe 
    if ($classe_turma_norm === '10ª') {
        
        $nomes_oficiais = [
            'L.Port', 'Inglês', 'Francês', 'Língua Nacional', 'Mat', 'Física', 'Biologia',
            'Química', 'História', 'Geografia', 'Informática', 'Exp. Musical', 'Exp. Plást',
            'Exp. Motora', 'Psicologia',
        ];

    // 2. REGRA PARA 11ª CLASSE INFÂNCIA
    } elseif ($classe_turma_norm === '11ª' && $curso_norm === normalizar_nome(CURSO_INFANCIA_NORM)) {
        
        $nomes_oficiais = [
            'L.Port', 'Inglês', 'Francês', 'L.Nacional', 'Mat', 'Física', 'Biologia',
            'Química', 'História', 'Geografia', 'Informática', 'Exp.Musical', 'Exp.Plást',
            'Exp.Motora', 'Psicologia', 'Filosofia', 'T.E.D.C', 'N.E.E',
            'Met.L.Infantil', 'Met.L.Nacional', 'Met.Est.Meio', 'P.P.S1',
            'ASE & AGE', 'F.P.S.D', 'H.S.E', 'Met.Exp.Motora', 'Met.Exp.Plást',
            'Met.Mat', 'Met.Exp.Musical', 'P.P.S2'
        ];
        
    // 3. REGRA PARA OUTRAS CLASSES SECUNDÁRIAS (12ª Infância, 11ª Primário, etc)
    } elseif (in_array($classe_turma_norm, CLASSES_SECUNDARIAS) && 
              ($curso_norm === normalizar_nome(CURSO_PRIMARIO_NORM) || $curso_norm === normalizar_nome(CURSO_INFANCIA_NORM))) {
        
        $nomes_oficiais = [
            'L.Port', 'Inglês', 'Francês', 'Língua Nacional', 'Mat', 'Física', 'Biologia',
            'Química', 'História', 'Geografia', 'Informática', 'Exp. Musical', 'Exp. Plást',
            'Exp. Motora', 'Psicologia', 'Filosofia', 'T.E.D.C', 'N.E.E', 'Met.L.Port',
            'Met.L.Nacional', 'Met.Inglês', 'Met.Est.Meio/C.N', 'F.P.S.D', 'H.S.E',
            'ASE e AGE', 'Met.Mat', 'Met.Hist/Geog', 'Met.Expressões', 'Met.L.Franc',
            'P.P.S1', 'P.P.S2'
        ];
    } else {
        $nomes_oficiais = [];
    }
    
    foreach ($nomes_oficiais as $i => $n) {
        // IDs negativos são usados como placeholders
        $disciplinas_para_ordenar[] = ['id' => -($i + 1), 'nome' => corrigir_acentos($n)];
    }
} else {
    // SE HÁ DISCIPLINAS NO BD, USA SOMENTE O RESULTADO DO BD (CONFORME SOLICITADO)
    $disciplinas_para_ordenar = $res_disciplinas_bd;
}
    
    // 4.3. Filtra duplicatas e ordena a lista
    $disciplinas_unicas = [];
    $nomes_normalizados_vistos = [];

    foreach ($disciplinas_para_ordenar as $disc) {
        $nome_norm_comparacao = normalizar_nome($disc['nome']);
        
        if (!isset($nomes_normalizados_vistos[$nome_norm_comparacao])) {
            $disc['nome'] = corrigir_acentos($disc['nome']); 
            $disciplinas_unicas[] = $disc;
            $nomes_normalizados_vistos[$nome_norm_comparacao] = true;
        }
    }
    
    // Ordena (e agora, NÃO FILTRA/EXCLUI)
    $disciplinas = ordenar_disciplinas($disciplinas_unicas, $classe_turma, $curso_turma);

    // --- 4.4. Busca de Todas as Notas (Apenas se houver IDs reais)
    // O MAPA DE NOTAS SÓ É MONTADO COM DISCIPLINAS DO BD (disc['id'] > 0)
    if ($disciplinas_com_id_real) {
        $sql_todas_notas = "
            SELECT aluno_id, disciplina_id, mt
            FROM notas
            WHERE turma_id = ? AND trimestre = ?
        ";
        $stmt_todas_notas = $conn->prepare($sql_todas_notas);
        $stmt_todas_notas->bind_param("ii", $turma_id, $trimestre);
        $stmt_todas_notas->execute();
        $res_todas_notas = $stmt_todas_notas->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt_todas_notas->close();

        foreach ($res_todas_notas as $nota) {
            $mapa_notas[$nota['aluno_id']][$nota['disciplina_id']] = $nota['mt'];
        }
    }

    // --- 4.5. MONTAGEM DA TABELA DE DADOS ---
    foreach ($alunos as $i => $aluno) {
        $linha = ['n'=>$i+1,'nome'=>$aluno['nome'],'notas'=>[],'obs'=>''];
        $disciplinas_negativas = 0; // Contador de negativas
        $total_disciplinas_com_nota = 0;

        foreach ($disciplinas as $disc) {
            $media_formatada = null;
            // Usa o ID da disciplina (real ou placeholder negativo)
            $mt_valor = $mapa_notas[$aluno['id']][$disc['id']] ?? null;

            if ($mt_valor !== null && $mt_valor !== '') {
                $media_formatada = number_format(floatval($mt_valor), 2, '.', ''); 
                $media_numerica = floatval($mt_valor);
            } else $media_numerica = 0;

            // Critério de Aproveitamento (Conta apenas se for uma disciplina com ID real)
            if ($disc['id'] > 0) {
                if ($media_numerica > 0) {
                    $total_disciplinas_com_nota++;
                    // MODIFICAÇÃO: Conta a nota negativa
                    if ($media_numerica < 10) $disciplinas_negativas++;
                }
            }

            $linha['notas'][$disc['id']] = $media_formatada;
        }

        // Determinação da Observação
        if (!$disciplinas_com_id_real) {
            $linha['obs'] = ''; // Não pode avaliar se não há IDs de disciplinas
        } elseif ($total_disciplinas_com_nota == 0) {
            $linha['obs'] = 'Sem Notas'; // Se não há notas lançadas em nenhuma disciplina
        } elseif ($disciplinas_negativas > 0) {
            $linha['obs'] = $disciplinas_negativas; // Se houver notas negativas, exibe o número
        } else {
            $linha['obs'] = 'Excelente'; // Se não houver notas negativas
        }

        $dados[] = $linha;
    }
}

$data_atual = date('d/m/Y');
if ($conn) $conn->close();
?>
<!DOCTYPE html>
<html lang="pt">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Mapa de Aproveitamento</title>
<style>
/* === ESTILOS GERAIS === */ 
body { font-family: 'Segoe UI', sans-serif; margin: 0; background-color: #f4f7f9; color: #333; display: flex; justify-content: center; flex-direction: column; align-items: center; padding: 10px; box-sizing: border-box; } 
.container { max-width: 95%; width: 100%; background-color: #ffffff; padding: 15px; box-shadow: 0 2px 7px rgba(0, 0, 0, 0.1); border-radius: 5px; overflow: hidden; position: relative; } 
/* === FORMULÁRIO E CABEÇALHO === */ 
.header-form-container { display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; margin-bottom: 10px; gap: 10px; flex-direction: row-reverse; } 
form { display: flex; align-items: center; flex-wrap: wrap; gap: 8px; } 
select, button { padding: 6px 10px; border: 1px solid #ccc; border-radius: 3px; font-size: 12px; transition: all 0.3s ease; } 
button { background-color: #007bff; color: white; cursor: pointer; border: none; } 
button:hover { background-color: #0056b3; } 
#print-button { background-color: #6c757d; } 
#print-button:hover { background-color: #5a6268; } 

/* 🌟 ALTERAÇÃO: AUMENTO DO TAMANHO DE LETRA DO CABEÇALHO (TELA) */
.cabecalho { text-align: center; margin-bottom: 15px; line-height: 1.1; } 
.cabecalho img { width: 70px; height: auto; margin-bottom: 5px; } 
h2 { margin: 0; font-weight: 800; font-size: **1.5em**; } /* Aumentado */ 
h3 { margin: 0; font-weight: 600; font-size: **1.1em**; } /* Aumentado */ 
.info-trimestre-turma { text-align: center; margin-bottom: 10px; font-size: **1.1em**; font-weight: bold; } /* Aumentado */

/* 🌟 ALTERAÇÃO: AUMENTO DO TAMANHO DE LETRA DO VISTO DO DIRETOR (TELA) */
.visto-director { text-align: center; font-size: **1.0em**; position: relative; } /* Aumentado */ 
.visto-director .line { border-bottom: 1px dotted #000; width: 150px; margin: 5px auto; } 
.visto-director .director-nome { margin-top: 5px; font-weight: bold; } 

/* === TABELA === */ 
.table-container { overflow-x: auto; width: 100%; } 
table { 
    border-collapse: collapse; 
    width: 100%; 
    font-size: 11px;
    table-layout: fixed; 
} 
th, td { 
    border: 1px solid #e0e0e0; 
    padding: 6px 5px;
    height: 35px;
    text-align: center; 
} 
/* Fonte do nome do aluno (2ª coluna) */
td:nth-child(2) {
    font-size: 11px; 
    text-align: left;
    word-break: break-word;
}

thead th { background-color: #003366; color: white; font-weight: 600; vertical-align: middle; } 
tbody tr:nth-child(even) { background-color: #f9f9f9; } 
tbody tr:hover { background-color: #f1f1f1; } 
td.vermelho { color: #d9534f; font-weight: bold; } 
/* === LARGURA DE COLUNAS (TELA) === */ 
<?php 
$num_disciplinas = max(1, count($disciplinas)); 
// Largura fixa (Nº: 4% + Nome: 18% + Obs: 6%) = 28%
$largura_fixa_total_tela = 4 + 18 + 6;
$livre_tela_percent = 100 - $largura_fixa_total_tela;
$largura_disciplina_percent = $livre_tela_percent / $num_disciplinas; 
$largura_disciplina_string = number_format($largura_disciplina_percent, 2) . '%'; ?> 
th:nth-child(1), td:nth-child(1) { width: 4%; } /* Nº */ 
th:nth-child(2), td:nth-child(2) { width: 18%; text-align: left; word-break: break-word; } /* Nome */ 
th.disciplina, td.disciplina { width: <?= $largura_disciplina_string ?>; word-break: break-word; white-space: normal; } /* Disciplina */ 
th:last-child, td:last-child { width: 6%; } /* Obs. */ 
/* === RODAPÉ: ASSINATURAS E DATA === */ 
.assinaturas { width: 100%; display: flex; justify-content: space-between; margin-top: 25px; font-size: 0.9em; } 
.assinatura-esquerda, .assinatura-direita { text-align: center; width: 45%; } 
.assinatura-esquerda .line, .assinatura-direita .line { padding-top: 50px; border-top: 1px dashed #000; } 
.data-mapa { text-align: right; margin-top: 20px; font-size: 0.9em; } 
/* ============================================ */ 
/* ============================================ */
/* ESTILOS PARA IMPRESSÃO A4 VERTICAL (PORTRAIT) */
@media print {
    @page { size: A4 portrait; margin: 2mm; }

    body {
        background-color: #fff;
        padding: 0;
        zoom: 90%;
        -webkit-print-color-adjust: exact !important;
        color-adjust: exact !important;
        font-family: "Arial", sans-serif;
    }

    .header-form-container, #print-button, button { display: none !important; }
    .container { box-shadow: none; padding: 0; max-width: 100%; font-size: 7px; }

    /* CABEÇALHO */
    .cabecalho { margin-bottom: 5px; line-height: 1; font-weight: bold; }
    .cabecalho img { width: 40px; }
    h2 { font-size: 1em; font-weight: bold !important; }
    h3 { font-size: 0.85em; font-weight: bold !important; }
    .info-trimestre-turma { font-size: 0.75em; margin-bottom: 2px; font-weight: bold !important; }

    /* VISTO DO DIRETOR */
    .visto-director { font-size: 0.7em; margin-top: 3px; font-weight: bold; }
    .visto-director .line { width: 100px; margin: 3px auto 0; }
    .visto-director .director-nome { margin-top: 1px; font-weight: bold; }

    /* TABELA */
    table {
        width: 100%;
        border-collapse: collapse;
        font-size: 6.5px; /* fonte menor para caber 40 alunos */
        table-layout: fixed;
        margin: 0;
    }
    th, td {
        border: 1px solid #000;
        padding: 1px 1px;  /* padding reduzido */
        height: 12px;      /* altura da linha menor */
        white-space: normal !important;
        text-align: center;
        word-wrap: break-word;
    }

    /* Nº e Nome do Aluno */
    th:nth-child(1), td:nth-child(1) { width: 2%; font-weight: bold; }  /* Nº */
    th:nth-child(2), td:nth-child(2) { width: 15%; text-align: left; padding-left: 2px; font-weight: bold; }

    /* Colunas de notas dinâmicas */
    th.disciplina, td.disciplina {
        width: calc((100% - 2% - 15% - 1.5%) / 17); /* ajusta para 17 disciplinas */
        font-size: 5.5px;
        line-height: 1;
    }

    /* Coluna Obs reduzida */
    th:last-child, td:last-child { width: 1.5%; font-size: 5.5px; }

    /* ASSINATURAS */
    .assinaturas { 
        margin-top: 5px; 
        font-size: 0.9em;
        font-weight: bold;    
        display: flex;
        justify-content: space-between;
    }
    .assinatura-esquerda, .assinatura-direita {
        text-align: center;
        width: 45%;
    }
    .assinatura-esquerda .line, .assinatura-direita .line {
        border-bottom: 2px solid #000; 
        height: 35px;                  /* espaço menor para assinar */
        margin-top: 5px;
    }
    .assinatura-esquerda .nome, .assinatura-direita .nome {
        margin-top: 2px;
        font-weight: bold;
        font-size: 0.85em;
    }

    .data-mapa { 
        font-size: 0.85em; 
        margin-top: 2px;
        font-weight: bold;
    }
}

</style>
</head>
<body>

<div class="container">
    <div class="header-form-container">
        <div class="visto-director">
            A DIRECTORA
            <div class="line"></div>
            <div class="director-nome">LIC. Carolina Patrícia</div>
        </div>
        <form method="get">
            <label style="font-size: 0.9em;">Turma:</label>
            <select name="turma_id" required style="font-size: 0.9em;">
                <option value="">-- Escolher Turma --</option>
                <?php foreach ($turmas as $t): ?>
                <option value="<?= $t['id'] ?>" <?= ($t['id'] == $turma_id ? 'selected' : '') ?>>
                    <?= htmlspecialchars("{$t['classe']} - {$t['nome']} - {$t['curso']}") ?>
                </option>
                <?php endforeach; ?>
            </select>
            <label style="font-size: 0.9em;">Trimestre:</label>
            <select name="trimestre" required style="font-size: 0.9em;">
                <option value="1" <?= $trimestre == 1 ? 'selected' : '' ?>>I</option>
                <option value="2" <?= $trimestre == 2 ? 'selected' : '' ?>>II</option>
                <option value="3" <?= $trimestre == 3 ? 'selected' : '' ?>>III</option>
            </select>
            <button type="submit" style="font-size: 0.9em;">Ver Mapa</button>
            <button id="print-button" type="button" onclick="window.print()" style="font-size: 0.9em;">Imprimir</button>
        </form>
    </div>

    <?php if ($turma_id && $trimestre): ?>
        <?php if (empty($alunos) && count($disciplinas) === 0): ?>
            <div style="text-align: center; margin-top: 20px; font-size: 1.1em; color: #cc3300;">
                Nenhum aluno encontrado e não foi possível carregar a lista de disciplinas para esta turma/curso.
            </div>
        <?php else: ?>
            <div class="cabecalho">
                <img src="insignia/insignia.png" alt="Insígnia Oficial de Angola" /> 
                <h3>REPÚBLICA DE ANGOLA</h3>
                <h3>MINISTÉRIO DA EDUCAÇÃO</h3>
                <h3>MAGISTÉRIO Nº 0038 BLA - 5 DE OUTUBRO - GAMA - CATUMBELA</h3>
                <h4>MAPA DE APROVEITAMENTO ESCOLAR</h4>
            </div>
            <div class="info-trimestre-turma">
                <?= htmlspecialchars($trimestre_extenso) ?> - Turma: <?= htmlspecialchars($classe_turma) ?> - <?= htmlspecialchars($nome_turma) ?> - <?= htmlspecialchars($curso_turma) ?>
            </div>

            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th rowspan="2">Nº</th>
                            <th rowspan="2">Nome do Aluno</th>
                            <th colspan="<?= count($disciplinas) ?>">Disciplinas</th>
                            <th rowspan="2">Obs.</th>
                        </tr>
                        <tr>
                            <?php foreach ($disciplinas as $d): ?>
                                <th class="disciplina"><?= htmlspecialchars($d['nome']) ?></th>
                            <?php endforeach; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($dados as $linha): ?>
                            <tr>
                                <td><?= $linha['n'] ?></td>
                                <td><?= htmlspecialchars($linha['nome']) ?></td>
                                <?php foreach ($disciplinas as $d): 
                                    $nota_display = $linha['notas'][$d['id']] ?? null;
                                    
                                    if ($nota_display === null) { 
                                        $display_nota = '00,00'; 
                                        $nota_numerica_para_cor = 0;
                                    } else {
                                        // Substitui o ponto por vírgula para exibição no padrão PT
                                        $display_nota = str_replace('.', ',', $nota_display); 
                                        $nota_numerica_para_cor = floatval($nota_display);
                                    }
                                ?>
                                <td class="<?= $nota_numerica_para_cor > 0 && $nota_numerica_para_cor < 10 ? 'vermelho' : '' ?>">
                                    <?= $display_nota ?>
                                </td>
                                <?php endforeach; ?>
                                <?php
                                // Lógica para aplicar a cor vermelha na Observação
                                $is_vermelho_obs = ($linha['obs'] === 'Sem Notas') || (is_numeric($linha['obs']) && $linha['obs'] > 0);
                                ?>
                                <td class="<?= $is_vermelho_obs ? 'vermelho' : '' ?>">
                                    <?= htmlspecialchars($linha['obs']) ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <div class="data-mapa">
                Catumbela, aos <?= $data_atual ?>
            </div>
            <div class="assinaturas">
                <div class="assinatura-esquerda">
                    <p>o Director de Turma</p>
                    <div class="line"></div>
                </div>
                <div class="assinatura-direita">
                    <p> O Subdirector Pedagógico</p>
                    <div class="line"></div>
                </div>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Lógica de quebra de página para impressão (útil para relatórios longos)
    if (window.matchMedia('print').matches) {
        const linhas = document.querySelectorAll('table tbody tr');
        
        // Número ideal de linhas para ocupar a folha com a altura de 13px
        const maxLinhas = 40; 
        
        if (linhas.length > maxLinhas) {
            const tabelaOriginal = document.querySelector('table');
            const cabecalho = tabelaOriginal.querySelector('thead').cloneNode(true);
            let linhas_por_pagina = 0;
            let linhasRemovidas = [];
            let corpoAtual = tabelaOriginal.querySelector('tbody');
            
            // Função para criar uma nova tabela e forçar quebra de página
            const criarNovaTabela = () => {
                const container = document.querySelector('.container');
                
                // Cria uma div para forçar a quebra de página
                const pageBreakDiv = document.createElement('div');
                pageBreakDiv.style.pageBreakBefore = 'always';
                pageBreakDiv.style.marginTop = '15px'; 
                container.appendChild(pageBreakDiv);

                // Clonar cabeçalho e info da turma para a nova página
                const cabecalhoClone = document.querySelector('.cabecalho').cloneNode(true);
                const infoClone = document.querySelector('.info-trimestre-turma').cloneNode(true);
                pageBreakDiv.appendChild(cabecalhoClone);
                pageBreakDiv.appendChild(infoClone);
                
                // Cria novo container de tabela e a tabela com cabeçalho
                const novaTabelaContainer = document.createElement('div');
                novaTabelaContainer.className = 'table-container';
                
                const novaTabela = tabelaOriginal.cloneNode(false);
                novaTabela.appendChild(cabecalho.cloneNode(true));
                
                // Cria e retorna o novo corpo para onde as linhas serão movidas
                const novoCorpo = document.createElement('tbody');
                novaTabela.appendChild(novoCorpo);
                novaTabelaContainer.appendChild(novaTabela);
                pageBreakDiv.appendChild(novaTabelaContainer); 
                
                return novoCorpo;
            };

            
            // Move as linhas excedentes para novas páginas
            for (let i = maxLinhas; i < linhas.length; i++) {
                if (i === maxLinhas || linhas_por_pagina >= maxLinhas) {
                    corpoAtual = criarNovaTabela();
                    linhas_por_pagina = 0;
                }
                
                // Move a linha original para o novo corpo
                const linhaClone = linhas[i].cloneNode(true);
                corpoAtual.appendChild(linhaClone);
                linhasRemovidas.push(linhas[i]);
                
                linhas_por_pagina++;
            }
            
            // Remove as linhas originais excedentes (mantendo apenas as primeiras maxLinhas)
            linhasRemovidas.forEach(linha => linha.remove());
        }
    }
});
</script>

</body>
</html>